package ai.connectif.sdk

import ai.connectif.sdk.data.model.push.TrackPushClickData
import ai.connectif.sdk.manager.Logger
import ai.connectif.sdk.model.Cart
import ai.connectif.sdk.model.ContactInfo
import ai.connectif.sdk.model.EventCallbacks
import ai.connectif.sdk.model.Product
import ai.connectif.sdk.model.Purchase
import ai.connectif.sdk.model.RegisterContactInfo
import ai.connectif.sdk.model.customevent.CustomEvent
import android.content.Context

object Connectif : ConnectifInterface {

    internal var connectifInternal = ConnectifInternal()

    /**
     * Initializes the Connectif SDK with the given parameters.
     *
     * @param context The Android context.
     * @param apiKey The API key provided by Connectif.
     * @param email (Optional) The email address associated with the contact.
     * @param connectifConfig [ConnectifConfig] Additional configuration parameters for the SDK.
     *
     * This method sets up the SDK and prepares it for handling events, push notifications, and contact-related data.
     */
    override fun initialize(
        context: Context,
        apiKey: String,
        email: String?,
        connectifConfig: ConnectifConfig
    ) {
        try {
            connectifInternal.initialize(context, apiKey, email, connectifConfig)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Adds a Firebase push token to the Connectif system.
     *
     * @param token The push notification token.
     * @param context The Android context.
     *
     * Call this method after retrieving the push notification token from Firebase.
     */
    override fun addPushToken(token: String, context: Context) {
        try {
            connectifInternal.addPushToken(token, context)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Handles incoming push notifications from Connectif.
     *
     * @param data A map containing the notification payload data.
     * @param context The Android context.
     * @return `true` if the notification was recognized as a Connectif notification and handled, `false` otherwise.
     *
     * This method should be called in your FCM messaging service to handle notifications sent by Connectif.
     */
    override fun handlePushNotification(data: Map<String, String>, context: Context): Boolean =
        try {
            connectifInternal.handlePushNotification(data, context)
        } catch (e: Exception) {
            handleInternalError(e)
            false
        }

    /**
     * Determines if the provided data corresponds to a Connectif push notification.
     *
     * @param data A map containing the notification payload data.
     * @return `true` if the data represents a Connectif push notification, `false` otherwise.
     */
    override fun isConnectifPushNotification(data: Map<String, String>?): Boolean = try {
        connectifInternal.isConnectifPushNotification(data)
    } catch (e: Exception) {
        handleInternalError(e)
        false
    }

    /**
     * Sends a page visit event to Connectif.
     *
     * @param name The name or identifier of the visited page.
     * @param callbacks Optional callbacks to handle success/failure events.
     */
    override fun sendPageVisit(name: String, callbacks: EventCallbacks?) {
        try {
            connectifInternal.sendPageVisit(name, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends a search event to Connectif.
     *
     * @param searchText The text that was searched.
     * @param callbacks Optional callbacks to handle success/failure events.
     */
    override fun sendSearch(searchText: String, callbacks: EventCallbacks?) {
        try {
            connectifInternal.sendSearch(searchText, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends a product visit event to Connectif.
     *
     * @param product The product that was visited, including its identifier and other relevant data.
     * @param callbacks Optional callbacks to handle success/failure events.
     */
    override fun sendProductVisit(product: Product, callbacks: EventCallbacks?) {
        try {
            connectifInternal.sendProductVisit(product, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends a login event to Connectif.
     *
     * @param email The email address of the contact who logged in.
     * @param callbacks Optional callbacks to handle success/failure events.
     *
     * Use this event to notify Connectif that a contact has identified themselves.
     *
     * Internally, if a push token is cached, it will be sent to Connectif alongside this login event.
     *
     */
    override fun sendLogin(email: String, callbacks: EventCallbacks?) {
        try {
            connectifInternal.sendLogin(email, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends a cart event to Connectif.
     *
     * @param cart The current state of the contact's cart, including products and prices.
     * @param callbacks Optional callbacks to handle success/failure events.
     *
     * This event helps Connectif understand the contact's ongoing journey and cart state.
     */
    override fun sendCart(cart: Cart, callbacks: EventCallbacks?) {
        try {
            connectifInternal.sendCart(cart, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends a purchase event to Connectif.
     *
     * @param purchase The completed purchase details, including products, total amount, and order ID.
     * @param callbacks Optional callbacks to handle success/failure events.
     *
     * This event should be sent after a successful checkout.
     */
    override fun sendPurchase(purchase: Purchase, callbacks: EventCallbacks?) {
        try {
            connectifInternal.sendPurchase(purchase, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends a registration event to Connectif.
     *
     * @param email The email address of the newly registered contact.
     * @param contactInfo Optional additional contact information for the contact.
     * @param callbacks Optional callbacks to handle success/failure events.
     *
     * Notify Connectif of new contact registrations to create their profile.
     *
     * Internally, if a push token is stored, it will be sent to Connectif alongside this registration event.
     */
    override fun sendRegister(
        email: String,
        contactInfo: RegisterContactInfo?,
        callbacks: EventCallbacks?
    ) {
        try {
            connectifInternal.sendRegister(email, contactInfo, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends updated contact information to Connectif.
     *
     * @param contactInfo The updated contact information.
     * @param callbacks Optional callbacks to handle success/failure events.
     *
     * Keep contact profiles accurate by informing Connectif of changes in their contact details.
     */
    override fun sendContactInfo(contactInfo: ContactInfo, callbacks: EventCallbacks?) {
        try {
            connectifInternal.sendContactInfo(contactInfo, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Subscribes the contact to a newsletter via Connectif.
     *
     * @param callbacks Optional callbacks to handle success/failure events.
     *
     * Use this to register the contact's interest in receiving newsletters.
     */
    override fun subscribeToNewsletter(callbacks: EventCallbacks?) {
        try {
            connectifInternal.subscribeToNewsletter(callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends a custom event to Connectif using an event ID.
     *
     * @param eventId The unique identifier for the custom event.
     * @param customEvent The custom event details.
     * @param callbacks Optional callbacks to handle success/failure events.
     *
     * Custom events allow you to extend the data you send to Connectif beyond the predefined schemas.
     */
    override fun sendCustomEventById(
        eventId: String,
        customEvent: CustomEvent,
        callbacks: EventCallbacks?
    ) {
        try {
            connectifInternal.sendCustomEventById(eventId, customEvent, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends a custom event to Connectif using an event ID and a generic payload.
     *
     * @param eventId The unique identifier for the custom event.
     * @param payload The payload representing the event data. It can be any object.
     * @param callbacks Optional callbacks to handle success/failure events.
     *
     * This method offers flexibility to provide a custom payload instead of a typed [CustomEvent].
     */
    override fun sendCustomEventById(eventId: String, payload: Any?, callbacks: EventCallbacks?) {
        try {
            connectifInternal.sendCustomEventById(eventId, payload, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends a custom event to Connectif using an event alias.
     *
     * @param eventAlias The alias (human-readable name) for the custom event.
     * @param customEvent The custom event details.
     * @param callbacks Optional callbacks to handle success/failure events.
     *
     * This variant allows referencing a custom event by an alias rather.
     */
    override fun sendCustomEventByAlias(
        eventAlias: String,
        customEvent: CustomEvent,
        callbacks: EventCallbacks?
    ) {
        try {
            connectifInternal.sendCustomEventByAlias(eventAlias, customEvent, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Sends a custom event to Connectif using an event alias and a generic payload.
     *
     * @param eventAlias The alias for the custom event.
     * @param payload The payload representing the event data. It can be any object.
     * @param callbacks Optional callbacks to handle success/failure events.
     *
     * Like [sendCustomEventById] with payload, but referenced by an alias.
     */
    override fun sendCustomEventByAlias(
        eventAlias: String,
        payload: Any?,
        callbacks: EventCallbacks?
    ) {
        try {
            connectifInternal.sendCustomEventByAlias(eventAlias, payload, callbacks)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Tracks when a contact interacts with a push notification.
     *
     * @param trackPushClickData The data associated with the push notification click event.
     *
     * This is used internally to record contact engagement with push notifications.
     */
    internal fun trackPushClick(trackPushClickData: TrackPushClickData?) {
        try {
            connectifInternal.trackPushClick(trackPushClickData)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    internal fun setTrackerId(tracker: String) {
        try {
            connectifInternal.setTrackerId(tracker)
        } catch (e: Exception) {
            handleInternalError(e)
        }
    }

    /**
     * Handles internal errors by logging them.
     *
     * @param e The exception that occurred.
     */
    private fun handleInternalError(e: Exception) {
        Logger.e("Error occurred: ${e.localizedMessage}")
    }
}
