package ai.connectif.sdk.data.source

import ai.connectif.sdk.Utils
import ai.connectif.sdk.data.error.ConnectifApiException
import ai.connectif.sdk.data.error.UnauthorizedApiException
import ai.connectif.sdk.data.source.model.request.GetNewTrackerRequest
import ai.connectif.sdk.manager.DeviceInfoManager
import android.content.SharedPreferences
import retrofit2.HttpException

internal class SettingLocalDataSourceImpl(
    private val deviceInfoManager: DeviceInfoManager,
    private val sharedPreferences: SharedPreferences
) : SettingLocalDataSource {
    private val keyTrackerId = "tracker_id"
    private val keyApiKey = "apiKey"
    private val keyEmail = "email"
    private val keyPushSmallIcon = "push_small_icon"
    private val keyPushChannelId = "push_channel_id"
    private val keyPushChannelName = "push_channel_name"
    private val keyDeviceId = "device_id"
    private val keyPushToken = "push_token"

    private suspend fun <T> safeApiCall(apiCall: suspend () -> T): T? = try {
        apiCall()
    } catch (e: HttpException) {
        when (e.code()) {
            401 -> throw UnauthorizedApiException("Invalid api key, please check it.", e)
            else -> throw ConnectifApiException(e.message(), e)
        }
    }

    override var trackerId: String?
        get() = sharedPreferences.getString(keyTrackerId, null)
        set(value) {
            sharedPreferences.edit().putString(keyTrackerId, value).apply()
        }

    override var apiKey: String?
        get() {
            return sharedPreferences.getString(keyApiKey, null)
        }
        set(value) {
            sharedPreferences.edit().putString(keyApiKey, value).apply()
        }

    override var deviceId: String?
        get() = sharedPreferences.getString(keyDeviceId, null)
        set(value) {
            sharedPreferences.edit().putString(keyDeviceId, value).apply()
        }
    override var pushToken: String?
        get() = sharedPreferences.getString(keyPushToken, null)
        set(value) {
            sharedPreferences.edit().putString(keyPushToken, value).apply()
        }

    override var email: String?
        get() = sharedPreferences.getString(keyEmail, null)
        set(value) {
            sharedPreferences.edit().putString(keyEmail, value).apply()
        }
    override var pushSmallIcon: Int
        get() = sharedPreferences.getInt(keyPushSmallIcon, 0)
        set(value) {
            sharedPreferences.edit().putInt(keyPushSmallIcon, value).apply()
        }

    override var pushChannelId: String?
        get() = sharedPreferences.getString(keyPushChannelId, null)
        set(value) {
            sharedPreferences.edit().putString(keyPushChannelId, value).apply()
        }
    override var pushChannelName: String?
        get() = sharedPreferences.getString(keyPushChannelName, null)
        set(value) {
            sharedPreferences.edit().putString(keyPushChannelName, value).apply()
        }

    override suspend fun getNewTrackerId(apiKey: String, deviceId: String): String? {
        val response = safeApiCall {
            RetrofitClient.getInstance().create(ConnectifApiService::class.java)
                .postGetNewTracker(
                    Utils.formatApiKeyToHeader(apiKey),
                    deviceId,
                    GetNewTrackerRequest(deviceInfoManager.getDeviceInfo(), email = email)
                )
        }
        return response?.trackerId
    }
}
