package a.yumi.lib.base.http

import a.yumi.lib.base.BaseApplication
import a.yumi.lib.base.util.log.loge
import android.content.Context
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Retrofit
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory
import retrofit2.converter.gson.GsonConverterFactory
import android.content.Context.CONNECTIVITY_SERVICE
import android.net.ConnectivityManager
import android.text.TextUtils
import java.io.BufferedReader
import java.io.DataOutputStream
import java.io.IOException
import java.io.InputStreamReader


object RetrofitUtil {
    fun get(url: String): Retrofit {
        // 初始化okhttp
        val networkMonitor = LiveNetworkMonitor(BaseApplication.context)

        val logging = HttpLoggingInterceptor()
        logging.level = HttpLoggingInterceptor.Level.BODY
        val client = OkHttpClient.Builder()
            .addInterceptor { chain ->
                val connected = networkMonitor.isConnected
                if (connected) {
                    return@addInterceptor chain.proceed(chain.request())
                } else {
                    throw NoNetworkException()
                }
            }
            .addNetworkInterceptor(logging)
            .build()

        // 初始化Retrofit
        return Retrofit.Builder()
            .client(client)
            .baseUrl(url)
            .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
            .addConverterFactory(GsonConverterFactory.create())
            .build()
    }
}

interface NetworkMonitor {
    val isConnected: Boolean
}

class LiveNetworkMonitor(context: Context) : NetworkMonitor {

    private val applicationContext: Context = context.applicationContext

    override val isConnected: Boolean
        get() {
            val realIp = if (TextUtils.isEmpty("")) "223.5.5.5" else ""
            val execCmd = execCmd(
                commands = arrayOf(String.format("ping -c 1 %s", realIp)),
                isRooted = false,
                isNeedResultMsg = true
            )
            loge(execCmd)
            return execCmd.result == 0
        }

    fun execCmd(
        commands: Array<String>?,
        isRooted: Boolean,
        isNeedResultMsg: Boolean
    ): CommandResult {
        var result = -1
        if (commands == null || commands.size == 0) {
            return CommandResult(result, "", "")
        }
        var process: Process? = null
        var successResult: BufferedReader? = null
        var errorResult: BufferedReader? = null
        var successMsg: StringBuilder? = null
        var errorMsg: StringBuilder? = null
        var os: DataOutputStream? = null
        try {
            process = Runtime.getRuntime().exec(if (isRooted) "su" else "sh")
            os = DataOutputStream(process!!.outputStream)
            for (command in commands) {
                if (command == null) continue
                os.write(command.toByteArray())
                os.writeBytes(LINE_SEP)
                os.flush()
            }
            os.writeBytes("exit$LINE_SEP")
            os.flush()
            result = process.waitFor()
            if (isNeedResultMsg) {
                successMsg = StringBuilder()
                errorMsg = StringBuilder()
                successResult = BufferedReader(
                    InputStreamReader(process.inputStream, "UTF-8")
                )
                errorResult = BufferedReader(
                    InputStreamReader(process.errorStream, "UTF-8")
                )
                var line: String? = successResult.readLine()
                if (line != null) {
                    successMsg.append(line)
                    line = successResult.readLine()
                    while (line != null) {
                        line = successResult.readLine()
                        successMsg.append(LINE_SEP).append(line)
                    }
                }
                line = errorResult.readLine()
                if (line != null) {
                    errorMsg.append(line)
                    line = errorResult.readLine()
                    while (line != null) {
                        line = errorResult.readLine()
                        errorMsg.append(LINE_SEP).append(line)
                    }
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
        } finally {
            try {
                os?.close()
            } catch (e: IOException) {
                e.printStackTrace()
            }

            try {
                successResult?.close()
            } catch (e: IOException) {
                e.printStackTrace()
            }

            try {
                errorResult?.close()
            } catch (e: IOException) {
                e.printStackTrace()
            }

            process?.destroy()
        }
        return CommandResult(
            result,
            successMsg?.toString() ?: "",
            errorMsg?.toString() ?: ""
        )
    }

}

class NoNetworkException : Exception("无网络连接")

class CommandResult(var result: Int, var successMsg: String, var errorMsg: String) {

    override fun toString(): String {
        return "result: " + result + "\n" +
                "successMsg: " + successMsg + "\n" +
                "errorMsg: " + errorMsg
    }
}

private val LINE_SEP = System.getProperty("line.separator")
