package a.yumi.jojo.lib.base.ui

import android.os.Bundle
import android.view.View
import com.jakewharton.rxbinding2.view.RxView
import com.trello.rxlifecycle2.components.support.RxFragment
import io.reactivex.Observable
import java.util.concurrent.TimeUnit

/**
 * PS:
 * Created by sdx on 2019/4/17.
 */
abstract class BaseFragment : RxFragment() {

    protected val TAG = this@BaseFragment::class.java.simpleName

    protected open var defaultWindowDuration: Long = 500 //默认点击间隔

    /**
     * 是否可见状态
     */
    private var isVisibled: Boolean = false

    /**
     * 标志位，View已经初始化完成
     */
    private var isPrepared: Boolean = false

    /**
     * 是否第一次加载
     */
    private var isFirstLoad = false

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        isFirstLoad = true
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        bindView(view, savedInstanceState)
        isPrepared = true
        lazyLoad()
    }

    /**
     * 如果是与ViewPager一起使用，调用的是setUserVisibleHint
     */
    override fun setUserVisibleHint(isVisibleToUser: Boolean) {
        super.setUserVisibleHint(isVisibleToUser)
        if (userVisibleHint) {
            isVisibled = true
            onVisible()
        } else {
            isVisibled = false
            onInvisible()
        }
    }

    /**
     * 如果是通过FragmentTransaction的show和hide的方法来控制显示，调用的是onHiddenChanged.
     * 若是初始就show的Fragment 为了触发该事件 需要先hide再show
     */
    override fun onHiddenChanged(hidden: Boolean) {
        super.onHiddenChanged(hidden)
        if (!hidden) {
            isVisibled = true
            onVisible()
        } else {
            isVisibled = false
            onInvisible()
        }
    }

    fun initData(view: View) {}

    protected fun onVisible() {
        lazyLoad()
    }

    protected fun onInvisible() {}

    /**
     * 要实现延迟加载Fragment内容,需要在 onCreate
     * isPrepared = true;
     */
    protected fun lazyLoad() {
        if (!isPrepared || !isVisible || !isFirstLoad) {
            return
        }
        isFirstLoad = false
        initData()
    }

    private fun bindClickEventOb(v: View): Observable<*> {
        return RxView.clicks(v)
            .throttleFirst(defaultWindowDuration, TimeUnit.MILLISECONDS)
            .compose(bindToLifecycle())
    }

    protected fun bindClickEvent(v: View) {
        bindClickEventOb(v).subscribe { onSingleClick(v) }
    }

    protected open fun onSingleClick(v: View) {}

    protected fun isFirstLoad(): Boolean {
        return isFirstLoad
    }

    fun refresh() {
        initData()
    }

    protected abstract fun bindView(view: View, bundle: Bundle?)

    protected abstract fun initData()
}